<?php
/**
 * Plugin Name: Sifet SEO
 * Description: Integra artigos da plataforma Sifet no WordPress via REST.
 * Version: 2.0.1.1
 * Author: Sifet Team
 * Author URI: https://sifet.com.br/
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Tested up to: 6.6
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: sifet
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SIFET_PLUGIN_VERSION', '2.0.1.1');
define('SIFET_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('SIFET_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SIFET_PLUGIN_BASENAME', plugin_basename(__FILE__));

function sifet_load_textdomain()
{
    load_plugin_textdomain('sifet', false, dirname(SIFET_PLUGIN_BASENAME) . '/languages/');
}
add_action('plugins_loaded', 'sifet_load_textdomain');

require_once SIFET_PLUGIN_PATH . 'includes/class-media.php';
require_once SIFET_PLUGIN_PATH . 'includes/class-seo.php';
require_once SIFET_PLUGIN_PATH . 'includes/rest-api.php';
require_once SIFET_PLUGIN_PATH . 'includes/admin.php';

/**
 * Inicializa área administrativa.
 */
function sifet_bootstrap_admin()
{
    static $admin_initialized = false;

    if ($admin_initialized || !is_admin()) {
        return;
    }

    $admin_initialized = true;
    $GLOBALS['sifet_admin'] = new Sifet_Admin();
}
add_action('plugins_loaded', 'sifet_bootstrap_admin', 9);

function sifet_bootstrap_rest_api()
{
    static $booted = false;

    if ($booted) {
        return;
    }

    $booted = true;

    // Gerenciador de mídia
    if (class_exists('Sifet_Media_Manager')) {
        $GLOBALS['sifet_media_manager'] = $GLOBALS['sifet_media_manager'] ?? new Sifet_Media_Manager();
    }

    // SEO manager para integrar com Rank Math / Yoast
    if (class_exists('Sifet_SEO_Manager')) {
        $GLOBALS['sifet_seo_manager'] = new Sifet_SEO_Manager();
    }

    if (class_exists('Sifet_REST_API')) {
        new Sifet_REST_API();
    }
}
add_action('plugins_loaded', 'sifet_bootstrap_rest_api', 20);

/**
 * Imprime JSON-LD de schema salvo no post (fora do <article>)
 */
function sifet_print_schema_jsonld()
{
    if (!is_singular('post')) {
        return;
    }

    $post_id = get_queried_object_id();
    if (!$post_id) {
        return;
    }

    // Buscar bundle salvo pelo endpoint
    $raw = get_post_meta($post_id, '_sifet_schema_bundle', true);

    if (empty($raw)) {
        $raw = get_post_meta($post_id, '_sifet_schema', true);
    }

    if (empty($raw)) {
        return;
    }

    $data = is_string($raw) ? json_decode($raw, true) : $raw;

    if (empty($data)) {
        return;
    }

    $json = wp_json_encode($data);
    if (empty($json)) {
        return;
    }

    echo "\n<!-- Sifet Schema JSON-LD -->\n<script type=\"application/ld+json\" data-sifet-schema=\"jsonld\">{$json}</script>\n";
}
add_action('wp_head', 'sifet_print_schema_jsonld', 5);

/**
 * Envia o token salvo para a API da Sifet para registrar o domínio.
 *
 * @param string|null $token Token opcional recebido pelo hook de atualização.
 * @return bool
 */
function sifet_send_token_to_nextjs($token = null)
{
    if ($token === null) {
        $token = get_option('token_sifet');
    }

    $token = trim((string) $token);
    $site_url = get_site_url();
    $nextjs_api_url = apply_filters('sifet_token_verification_endpoint', 'https://app.sifet.com.br/api/verify-token');

    if (empty($token) || empty($site_url)) {
        return false;
    }

    $args = array(
        'body'        => wp_json_encode(
            array(
                'token'   => $token,
                'siteUrl' => $site_url,
            )
        ),
        'headers'     => array(
            'Content-Type' => 'application/json',
            'Access-Token' => $token,
        ),
        'method'      => 'POST',
        'data_format' => 'body',
        'timeout'     => 15,
    );

    $response = wp_remote_post($nextjs_api_url, $args);

    if (is_wp_error($response)) {
        return false;
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);

    if ($status_code !== 200) {
        return false;
    }

    $data = json_decode($body, true);

    if (isset($data['error'])) {
        return false;
    }

    return true;
}

/**
 * Dispara a integração sempre que o token é atualizado.
 *
 * @param string $old_value Valor antigo.
 * @param string $value     Novo valor.
 */
function sifet_handle_token_update($old_value, $value)
{
    if (empty($value) || $value === $old_value) {
        return;
    }

    sifet_send_token_to_nextjs($value);
}
add_action('update_option_token_sifet', 'sifet_handle_token_update', 10, 2);

/**
 * Dispara quando o token é salvo pela primeira vez.
 *
 * @param string $option Nome da opção.
 * @param string $value  Valor salvo.
 */
function sifet_handle_token_add($option, $value)
{
    if ($option !== 'token_sifet' || empty($value)) {
        return;
    }

    sifet_send_token_to_nextjs($value);
}
add_action('add_option_token_sifet', 'sifet_handle_token_add', 10, 2);
