<?php
if (!defined('ABSPATH')) {
    exit;
}

class Sifet_REST_API
{
    const ROUTE_NAMESPACE = 'sifet/v1';

    private $allowed_statuses = array('publish', 'draft', 'future', 'pending', 'private');

    public function __construct()
    {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    public function register_routes()
    {
        register_rest_route(self::ROUTE_NAMESPACE, '/events', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_events'),
            'permission_callback' => array($this, 'authenticate'),
        ));

        register_rest_route(self::ROUTE_NAMESPACE, '/categories', array(
            'methods' => 'GET',
            'callback' => array($this, 'list_categories'),
            'permission_callback' => array($this, 'authenticate'),
        ));

        register_rest_route(self::ROUTE_NAMESPACE, '/categories', array(
            'methods' => 'POST',
            'callback' => array($this, 'create_categories'),
            'permission_callback' => array($this, 'authenticate'),
        ));

        register_rest_route(self::ROUTE_NAMESPACE, '/handshake', array(
            'methods' => 'GET',
            'callback' => array($this, 'handshake'),
            'permission_callback' => array($this, 'authenticate'),
        ));
    }

    public function authenticate($request)
    {
        $token = $this->extract_token($request);

        if (empty($token)) {
            return new WP_Error(
                'sifet_missing_token',
                __('Token de acesso não encontrado. Use Authorization: Bearer <token>.', 'sifet'),
                array('status' => 401)
            );
        }

        $stored_token = get_option('token_sifet');

        if (empty($stored_token)) {
            return new WP_Error(
                'sifet_token_unset',
                __('Token não configurado no WordPress. Defina em Configurações > Sifet.', 'sifet'),
                array('status' => 401)
            );
        }

        if (!hash_equals($stored_token, $token)) {
            return new WP_Error(
                'sifet_invalid_token',
                __('Token de acesso inválido.', 'sifet'),
                array('status' => 403)
            );
        }

        return true;
    }

    public function handle_events($request)
    {
        $event = strtoupper(sanitize_text_field($request->get_param('event')));

        if (empty($event)) {
            return $this->respond(400, __('Tipo de evento não especificado.', 'sifet'), array(), false);
        }

        if ($event === 'CHECK_INTEGRATION') {
            return $this->respond(200, __('Integração ativa.', 'sifet'), array(
                'token' => get_option('token_sifet'),
                'version' => defined('SIFET_PLUGIN_VERSION') ? SIFET_PLUGIN_VERSION : 'dev',
            ), true);
        }

        if ($event === 'POST_CREATED' || $event === 'POST_UPDATED') {
            return $this->handle_post_event($request);
        }

        return $this->respond(400, __('Evento não suportado.', 'sifet'), array('event' => $event), false);
    }

    public function handshake()
    {
        global $wp_version;

        return $this->respond(200, 'OK', array(
            'plugin_version' => defined('SIFET_PLUGIN_VERSION') ? SIFET_PLUGIN_VERSION : 'dev',
            'wp_version' => $wp_version,
            'features' => array(
                'posts' => true,
                'categories' => true,
                'featured_images' => true,
                'inline_images' => true,
                'seo_meta' => true,
            ),
        ), true);
    }

    public function list_categories()
    {
        $categories = get_terms(array(
            'taxonomy' => 'category',
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        ));

        if (is_wp_error($categories)) {
            return $this->respond(
                500,
                __('Erro ao buscar categorias.', 'sifet'),
                array('code' => $categories->get_error_code()),
                false
            );
        }

        $data = array_map(function ($cat) {
            return array(
                'id' => (int) $cat->term_id,
                'name' => $cat->name,
                'slug' => $cat->slug,
                'count' => (int) $cat->count,
            );
        }, $categories);

        $message = sprintf(
            /* translators: %d: number of categories found */
            _n('%d categoria encontrada.', '%d categorias encontradas.', count($data), 'sifet'),
            count($data)
        );

        return $this->respond(200, $message, array('categories' => $data), true);
    }

    public function create_categories($request)
    {
        $params = $request->get_json_params();
        if (!is_array($params)) {
            $params = array();
        }

        $categories = isset($params['categories']) ? (array) $params['categories'] : array();

        if (empty($categories)) {
            return $this->respond(
                400,
                __('Nenhuma categoria foi enviada para criação.', 'sifet'),
                array(),
                false
            );
        }

        $created = 0;
        $processed = array();
        $errors = array();

        foreach ($categories as $category) {
            $name = isset($category['name']) ? sanitize_text_field($category['name']) : '';
            $slug = sanitize_title($category['slug'] ?? $name);
            $description = sanitize_textarea_field($category['description'] ?? '');

            if (empty($name) && empty($slug)) {
                $errors[] = array(
                    'name' => $name,
                    'slug' => $slug,
                    'message' => __('Nome ou slug são obrigatórios para criar uma categoria.', 'sifet'),
                    'code' => 'sifet_category_missing_fields',
                );
                continue;
            }

            $existing = null;

            if (!empty($slug)) {
                $existing = get_term_by('slug', $slug, 'category');
            }

            if (!$existing && !empty($name)) {
                $existing = get_term_by('name', $name, 'category');
            }

            if ($existing && !is_wp_error($existing)) {
                if (!empty($description) && $existing->description !== $description) {
                    wp_update_term($existing->term_id, 'category', array('description' => $description));
                }

                $processed[] = array(
                    'term_id' => (int) $existing->term_id,
                    'name' => $existing->name,
                    'slug' => $existing->slug,
                    'created' => false,
                );
                continue;
            }

            $args = array();
            if (!empty($slug)) {
                $args['slug'] = $slug;
            }
            if (!empty($description)) {
                $args['description'] = $description;
            }

            $result = wp_insert_term($name ?: $slug, 'category', $args);

            if (is_wp_error($result)) {
                $errors[] = array(
                    'name' => $name,
                    'slug' => $slug,
                    'message' => $result->get_error_message(),
                    'code' => $result->get_error_code(),
                );
                continue;
            }

            $created++;
            $term_id = (int) $result['term_id'];
            $term = get_term($term_id, 'category');

            $processed[] = array(
                'term_id' => $term_id,
                'name' => ($term && !is_wp_error($term)) ? $term->name : ($name ?: $slug),
                'slug' => ($term && !is_wp_error($term)) ? $term->slug : ($slug ?: sanitize_title($name)),
                'created' => true,
            );
        }

        if ($created === 0 && empty($processed)) {
            $message = !empty($errors)
                ? $errors[0]['message']
                : __('Nenhuma categoria válida foi enviada.', 'sifet');

            return $this->respond(400, $message, array('errors' => $errors), false);
        }

        $message = $created > 0
            ? sprintf(
                _n('%d categoria criada.', '%d categorias criadas.', $created, 'sifet'),
                $created
            )
            : __('Nenhuma categoria nova foi criada. Todas já existem no WordPress.', 'sifet');

        $data = array(
            'created' => $created,
            'processed' => $processed,
        );

        if (!empty($errors)) {
            $data['errors'] = $errors;
        }

        return $this->respond(200, $message, $data, true);
    }

    private function handle_post_event($request)
    {
        $raw_post = $request->get_param('post');

        if (!is_array($raw_post)) {
            return $this->respond(400, __('Dados do post não fornecidos.', 'sifet'), array(), false);
        }

        $post = $this->normalize_post_payload($raw_post);
        if (is_wp_error($post)) {
            return $this->error_from_wp_error($post);
        }

        $categories = $this->ensure_categories($post['categories']);
        if (is_wp_error($categories)) {
            return $this->error_from_wp_error($categories);
        }

        $result = $this->upsert_post($post, $categories);
        if (is_wp_error($result)) {
            return $this->error_from_wp_error($result);
        }

        list($post_id, $created) = $result;

        return $this->respond(
            $created ? 201 : 200,
            $created ? __('Post criado com sucesso.', 'sifet') : __('Post atualizado com sucesso.', 'sifet'),
            array('post_id' => $post_id),
            true
        );
    }

    private function normalize_post_payload($post)
    {
        $id = sanitize_text_field($post['id'] ?? '');
        $slug = sanitize_title($post['slug'] ?? '');
        $title = sanitize_text_field($post['title'] ?? '');
        $content = $post['content_html'] ?? ($post['content']['html'] ?? ($post['content'] ?? ''));

        if (empty($id) || empty($slug) || empty($title) || empty($content)) {
            return new WP_Error(
                'sifet_missing_field',
                __('Campos obrigatórios: id, slug, title, content_html.', 'sifet'),
                array('status' => 400)
            );
        }

        $status = $this->normalize_status($post['status'] ?? 'publish');
        $description = sanitize_textarea_field($post['description'] ?? '');
        $publication_date = $this->normalize_date($post['publication_date'] ?? 'now');
        $content = $this->sanitize_html_content($content);

        $featured_image = array(
            'url' => esc_url_raw($post['featured_image']['url'] ?? ''),
            'filename' => sanitize_file_name($post['featured_image']['filename'] ?? ''),
            'alt_text' => sanitize_text_field($post['featured_image']['alt_text'] ?? $title),
        );

        $inline_images = array();
        if (!empty($post['images']) && is_array($post['images'])) {
            foreach ($post['images'] as $image) {
                $inline_images[] = array(
                    'url' => esc_url_raw($image['url'] ?? ''),
                    'filename' => sanitize_file_name($image['filename'] ?? ''),
                    'alt_text' => sanitize_text_field($image['alt_text'] ?? $title),
                );
            }
        }

        return array(
            'id' => $id,
            'slug' => $slug,
            'title' => $title,
            'status' => $status,
            'description' => $description,
            'publication_date' => $publication_date,
            'content' => $content,
            'featured_image' => $featured_image,
            'images' => $inline_images,
            'categories' => $this->normalize_categories($post['categories'] ?? array()),
            'seo' => is_array($post['seo'] ?? null) ? $post['seo'] : array(),
            'schema' => $post['schema'] ?? null,
            'main_keyword' => sanitize_text_field($post['main_keyword'] ?? ''),
        );
    }

    private function normalize_categories($categories)
    {
        $normalized = array();

        foreach ((array) $categories as $cat) {
            $name = isset($cat['name']) ? sanitize_text_field($cat['name']) : '';
            $slug = sanitize_title($cat['slug'] ?? $name);

            $normalized[] = array(
                'id' => isset($cat['id']) ? sanitize_text_field($cat['id']) : '',
                'name' => $name,
                'slug' => $slug,
                'create_if_missing' => array_key_exists('create_if_missing', $cat) ? (bool) $cat['create_if_missing'] : true,
            );
        }

        return $normalized;
    }

    private function ensure_categories($categories)
    {
        if (empty($categories)) {
            return new WP_Error(
                'sifet_missing_category',
                __('É necessário fornecer pelo menos uma categoria.', 'sifet'),
                array('status' => 400)
            );
        }

        $category_ids = array();

        foreach ($categories as $cat) {
            $term_id = null;

            if (!empty($cat['id'])) {
                $term_id = $this->find_category_by_meta($cat['id']);
            }

            if (!$term_id && !empty($cat['slug'])) {
                $term = get_term_by('slug', $cat['slug'], 'category');
                if ($term && !is_wp_error($term)) {
                    $term_id = $term->term_id;
                }
            }

            if (!$term_id && !empty($cat['name'])) {
                $term = get_term_by('name', $cat['name'], 'category');
                if ($term && !is_wp_error($term)) {
                    $term_id = $term->term_id;
                }
            }

            if ($term_id) {
                if (!empty($cat['id'])) {
                    update_term_meta($term_id, 'id_sifet', $cat['id']);
                }
                $category_ids[] = (int) $term_id;
                continue;
            }

            if (empty($cat['create_if_missing'])) {
                $label = $cat['name'] ?: $cat['slug'] ?: 'categoria';
                /* translators: %s category name */
                $message = sprintf(
                    __('Categoria "%s" não encontrada no sistema.', 'sifet'),
                    $label
                );

                return new WP_Error('sifet_category_not_found', $message, array('status' => 400));
            }

            $name = $cat['name'] ?: ucwords(str_replace('-', ' ', $cat['slug']));
            $args = !empty($cat['slug']) ? array('slug' => $cat['slug']) : array();

            $new_cat = wp_insert_term($name, 'category', $args);

            if (is_wp_error($new_cat)) {
                return $new_cat;
            }

            if (!empty($cat['id'])) {
                update_term_meta($new_cat['term_id'], 'id_sifet', $cat['id']);
            }

            $category_ids[] = (int) $new_cat['term_id'];
        }

        return array_values(array_unique($category_ids));
    }

    private function find_category_by_meta($id_sifet)
    {
        $terms = get_terms(array(
            'taxonomy' => 'category',
            'hide_empty' => false,
            'meta_query' => array(
                array(
                    'key' => 'id_sifet',
                    'value' => $id_sifet,
                ),
            ),
            'number' => 1,
        ));

        if (is_wp_error($terms) || empty($terms)) {
            return null;
        }

        return $terms[0]->term_id;
    }

    private function upsert_post($post, $category_ids)
    {
        $existing = $this->find_post_by_sifet_id($post['id']);
        $is_new = !$existing;

        $post_data = array(
            'ID' => $existing ?: 0,
            'post_type' => 'post',
            'post_status' => $post['status'],
            'post_title' => $post['title'],
            'post_name' => $post['slug'],
            'post_content' => $post['content'],
            'post_excerpt' => $post['description'],
            'post_date' => $post['publication_date'],
            'post_date_gmt' => gmdate('Y-m-d H:i:s', strtotime($post['publication_date'])),
            'meta_input' => array(
                'id_sifet' => $post['id'],
                '_sifet_main_keyword' => $post['main_keyword'],
            ),
        );

        $post_id = $existing
            ? wp_update_post($post_data, true)
            : wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return $post_id;
        }

        wp_set_post_categories($post_id, $category_ids);

        if (!empty($post['featured_image']['url'])) {
            $thumb_id = $this->import_image($post_id, $post['featured_image'], $post['title']);
            if (!is_wp_error($thumb_id)) {
                set_post_thumbnail($post_id, $thumb_id);
            }
        }

        $content_with_media = $this->replace_inline_images($post_id, $post['content'], $post['images'], $post['title']);

        if ($content_with_media !== $post['content']) {
            wp_update_post(array(
                'ID' => $post_id,
                'post_content' => $content_with_media,
            ));
        }

        $this->apply_seo_metadata($post_id, $post['seo'], $post['main_keyword']);
        $this->apply_schema_payload($post_id, $post['schema']);

        return array($post_id, $is_new);
    }

    private function find_post_by_sifet_id($id_sifet)
    {
        $posts = get_posts(array(
            'meta_key' => 'id_sifet',
            'meta_value' => $id_sifet,
            'post_type' => 'post',
            'post_status' => 'any',
            'numberposts' => 1,
        ));

        if (empty($posts)) {
            return null;
        }

        return (int) $posts[0]->ID;
    }

    private function import_image($post_id, $image, $fallback_alt = '')
    {
        $url = esc_url_raw($image['url'] ?? '');
        if (empty($url)) {
            return new WP_Error(
                'sifet_image_missing',
                __('URL da imagem não fornecida.', 'sifet'),
                array('status' => 400)
            );
        }

        if (!function_exists('media_sideload_image')) {
            require_once ABSPATH . 'wp-admin/includes/media.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        $alt = sanitize_text_field($image['alt_text'] ?? $fallback_alt);
        $filename = sanitize_file_name($image['filename'] ?? '');

        global $sifet_media_manager;

        if ($sifet_media_manager instanceof Sifet_Media_Manager) {
            $managed_upload = $sifet_media_manager->upload_image(
                $post_id,
                $url,
                $filename,
                $alt,
                $fallback_alt ?: $alt
            );

            if (!is_wp_error($managed_upload)) {
                return (int) $managed_upload;
            }
        }

        $attachment_id = media_sideload_image($url, $post_id, $alt, 'id');

        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        if (!empty($alt)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
        }

        if (!empty($filename)) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_title' => pathinfo($filename, PATHINFO_FILENAME),
            ));
        }

        update_post_meta($attachment_id, 'image_url_sifet', $url);

        return $attachment_id;
    }

    private function replace_inline_images($post_id, $content, $images, $fallback_alt)
    {
        if (empty($images) || !is_array($images)) {
            return $content;
        }

        $updated_content = $content;

        foreach ($images as $image) {
            if (empty($image['url'])) {
                continue;
            }

            $attachment_id = $this->import_image($post_id, $image, $fallback_alt);

            if (is_wp_error($attachment_id)) {
                continue;
            }

            $new_url = wp_get_attachment_url($attachment_id);

            if ($new_url) {
                $updated_content = str_replace($image['url'], $new_url, $updated_content);
            }
        }

        return $updated_content;
    }

    private function apply_seo_metadata($post_id, $seo, $focus_keyword = '')
    {
        if (empty($seo) || !is_array($seo)) {
            return;
        }

        $title = isset($seo['title']) ? sanitize_text_field($seo['title']) : '';
        $description = isset($seo['description']) ? sanitize_textarea_field($seo['description']) : '';
        $focus_keyword = sanitize_text_field($focus_keyword);

        if (!empty($title)) {
            update_post_meta($post_id, '_yoast_wpseo_title', $title);
            update_post_meta($post_id, '_rank_math_title', $title);
        }

        if (!empty($description)) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
            update_post_meta($post_id, '_rank_math_description', $description);
            update_post_meta($post_id, '_aioseop_description', $description);
        }

        // Palavras-chave principais para RankMath e Yoast
        if (!empty($focus_keyword)) {
            // Rank Math usa string ou array; aqui salvamos string simples
            update_post_meta($post_id, 'rank_math_focus_keyword', $focus_keyword);

            // Yoast (versões recentes) usa JSON com keyphrase + synonyms
            $yoast_focus = array(
                'keyphrase' => $focus_keyword,
                'synonyms' => '',
            );

            update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword); // fallback legado
            update_post_meta($post_id, '_yoast_wpseo_focuskeywords', wp_json_encode(array($yoast_focus)));
        }

        update_post_meta($post_id, '_sifet_seo_raw', wp_json_encode($seo));
    }

    /**
     * Aplica schemas (FAQ, HowTo, Review, Article) no post usando SEO Manager quando disponível
     */
    private function apply_schema_payload($post_id, $schema)
    {
        if (empty($schema)) {
            return;
        }

        $schemas = $this->normalize_schema_bundle($schema);

        if (empty($schemas)) {
            return;
        }

        // Guardar payload bruto para debug/fallback
        update_post_meta($post_id, '_sifet_schema', wp_json_encode($schema));
        update_post_meta($post_id, '_sifet_schema_bundle', wp_json_encode($schemas));

        global $sifet_seo_manager;

        foreach ($schemas as $schema_item) {
            if (!is_array($schema_item)) {
                continue;
            }

            $type_key = isset($schema_item['@type']) ? strtolower($schema_item['@type']) : 'generic';
            update_post_meta($post_id, "_sifet_schema_{$type_key}", wp_json_encode($schema_item));

            if ($sifet_seo_manager instanceof Sifet_SEO_Manager) {
                $sifet_seo_manager->apply_schema($post_id, $schema_item);
            }
        }
    }

    /**
     * Aceita tanto schema único quanto bundle com chaves (article/faq/howTo/review)
     */
    private function normalize_schema_bundle($schema_payload)
    {
        // Se já é um schema único
        if (is_array($schema_payload) && isset($schema_payload['@type'])) {
            return array($schema_payload);
        }

        $schemas = array();

        if (is_array($schema_payload)) {
            $keys = array('article', 'faq', 'howTo', 'review', 'product', 'schemas');

            foreach ($keys as $key) {
                if (!empty($schema_payload[$key])) {
                    if ($key === 'schemas' && is_array($schema_payload[$key])) {
                        foreach ($schema_payload[$key] as $item) {
                            if (is_array($item)) {
                                $schemas[] = $item;
                            }
                        }
                    } elseif (is_array($schema_payload[$key])) {
                        $schemas[] = $schema_payload[$key];
                    }
                }
            }
        }

        return $schemas;
    }

    /**
     * Anexa JSON-LD no final do conteúdo (ex: FAQ) para garantir schema mesmo se plugins não aplicarem
     */
    private function append_schema_jsonld($content, $schemas)
    {
        if (empty($schemas) || !is_array($schemas)) {
            return $content;
        }

        // Evitar duplicação
        if (stripos($content, 'data-sifet-schema="jsonld"') !== false) {
            return $content;
        }

        $json = wp_json_encode($schemas);
        if (empty($json)) {
            return $content;
        }

        $script = '<script type="application/ld+json" data-sifet-schema="jsonld">' . $json . '</script>';

        return $content . "\n\n" . $script;
    }

    private function sanitize_html_content($content)
    {
        $allowed = wp_kses_allowed_html('post');
        $allowed['iframe'] = array(
            'src' => true,
            'title' => true,
            'width' => true,
            'height' => true,
            'allow' => true,
            'allowfullscreen' => true,
            'loading' => true,
        );

        $allowed['img'] = array_merge($allowed['img'] ?? array(), array(
            'src' => true,
            'alt' => true,
            'title' => true,
            'class' => true,
            'loading' => true,
        ));

        return wp_kses(wp_unslash($content), $allowed);
    }

    private function normalize_status($status)
    {
        $status = strtolower(sanitize_text_field($status));

        $map = array(
            'scheduled' => 'future',
            'publicado' => 'publish',
            'published' => 'publish',
        );

        if (isset($map[$status])) {
            $status = $map[$status];
        }

        if (!in_array($status, $this->allowed_statuses, true)) {
            return 'publish';
        }

        return $status;
    }

    private function normalize_date($date_string)
    {
        $timestamp = strtotime($date_string);
        if ($timestamp === false) {
            $timestamp = time();
        }

        return gmdate('Y-m-d H:i:s', $timestamp);
    }

    private function extract_token($request)
    {
        $auth = $request->get_header('authorization');
        if (!empty($auth) && preg_match('/Bearer\s+(.*)$/i', $auth, $matches)) {
            return trim($matches[1]);
        }

        $fallback = $request->get_header('access-token');
        if (!empty($fallback)) {
            return trim($fallback);
        }

        $query = $request->get_param('token');
        if (!empty($query)) {
            return trim($query);
        }

        return null;
    }

    private function respond($status, $message, $data = array(), $success = null)
    {
        $payload = array('message' => $message);

        if ($success !== null) {
            $payload['success'] = $success;
        }

        if ($data !== null && $data !== array()) {
            $payload['data'] = $data;
        }

        return new WP_REST_Response($payload, $status);
    }

    private function error_from_wp_error(WP_Error $error)
    {
        $data = $error->get_error_data();
        $status = is_array($data) && isset($data['status']) ? $data['status'] : 400;

        return $this->respond($status, $error->get_error_message(), array('code' => $error->get_error_code()), false);
    }
}
