<?php
/**
 * Sifet SEO Manager
 * 
 * Gerenciamento de metadados SEO para os principais plugins:
 * - Yoast SEO
 * - Rank Math
 * - All in One SEO Pack
 * 
 * Suporte adicional:
 * - Open Graph (Facebook)
 * - Twitter Cards
 * - Canonical URLs
 * - Meta Robots
 * - Schema.org (FAQ, HowTo, Review)
 * 
 * @package Sifet
 * @version 1.0.8
 */

if (!defined('ABSPATH')) {
    exit;
}

class Sifet_SEO_Manager {
    
    /**
     * Plugins SEO detectados
     */
    private $active_seo_plugins = array();
    
    /**
     * Inicializar
     */
    public function __construct() {
        $this->detect_seo_plugins();
    }
    
    /**
     * Detectar plugins SEO ativos
     */
    private function detect_seo_plugins() {
        // Yoast SEO
        if (defined('WPSEO_VERSION') || class_exists('WPSEO_Options')) {
            $this->active_seo_plugins[] = 'yoast';
        }
        
        // Rank Math
        if (defined('RANK_MATH_VERSION') || class_exists('RankMath')) {
            $this->active_seo_plugins[] = 'rankmath';
        }
        
        // All in One SEO
        if (defined('AIOSEOP_VERSION') || class_exists('All_in_One_SEO_Pack')) {
            $this->active_seo_plugins[] = 'aioseo';
        }
        
        error_log('[Sifet SEO] Plugins detectados: ' . implode(', ', $this->active_seo_plugins));
    }
    
    /**
     * Aplicar metadados SEO a um post
     * 
     * @param int $post_id ID do post
     * @param array $seo_data Dados SEO do payload
     * @return bool True se aplicado com sucesso
     */
    public function apply_seo_metadata($post_id, $seo_data) {
        if (empty($seo_data) || !is_array($seo_data)) {
            return false;
        }
        
        $applied = false;
        
        // 1. Meta Title
        if (!empty($seo_data['title'])) {
            $applied = $this->set_meta_title($post_id, $seo_data['title']) || $applied;
        }
        
        // 2. Meta Description
        if (!empty($seo_data['meta_description'])) {
            $applied = $this->set_meta_description($post_id, $seo_data['meta_description']) || $applied;
        }
        
        // 3. Canonical URL
        if (!empty($seo_data['canonical'])) {
            $applied = $this->set_canonical($post_id, $seo_data['canonical']) || $applied;
        }
        
        // 4. Meta Robots
        if (!empty($seo_data['robots'])) {
            $applied = $this->set_robots($post_id, $seo_data['robots']) || $applied;
        }
        
        // 5. Open Graph
        if (!empty($seo_data['og']) && is_array($seo_data['og'])) {
            $applied = $this->set_open_graph($post_id, $seo_data['og']) || $applied;
        }
        
        // 6. Twitter Cards
        if (!empty($seo_data['twitter']) && is_array($seo_data['twitter'])) {
            $applied = $this->set_twitter_cards($post_id, $seo_data['twitter']) || $applied;
        }
        
        // 7. Focus Keyword (Yoast/Rank Math)
        if (!empty($seo_data['focus_keyword'])) {
            $applied = $this->set_focus_keyword($post_id, $seo_data['focus_keyword']) || $applied;
        }
        
        if ($applied) {
            error_log("[Sifet SEO] Metadados aplicados ao post {$post_id}");
        }
        
        return $applied;
    }
    
    /**
     * Aplicar Schema.org estruturado
     * 
     * @param int $post_id ID do post
     * @param array $schema_data Dados do schema
     * @return bool True se aplicado
     */
    public function apply_schema($post_id, $schema_data) {
        if (empty($schema_data) || !is_array($schema_data)) {
            return false;
        }
        
        $type = $schema_data['@type'] ?? '';
        
        switch ($type) {
            case 'FAQPage':
                return $this->set_faq_schema($post_id, $schema_data);
                
            case 'HowTo':
                return $this->set_howto_schema($post_id, $schema_data);
                
            case 'Review':
            case 'Product':
                return $this->set_review_schema($post_id, $schema_data);
                
            default:
                // Schema genérico
                return $this->set_generic_schema($post_id, $schema_data);
        }
    }
    
    /**
     * Definir meta title
     */
    private function set_meta_title($post_id, $title) {
        $title = sanitize_text_field($title);
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_yoast_wpseo_title', $title);
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            update_post_meta($post_id, 'rank_math_title', $title);
            $applied = true;
        }
        
        // All in One SEO
        if (in_array('aioseo', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_aioseop_title', $title);
            $applied = true;
        }
        
        // Fallback: meta genérico
        if (!$applied) {
            update_post_meta($post_id, '_sifet_seo_title', $title);
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir meta description
     */
    private function set_meta_description($post_id, $description) {
        $description = sanitize_textarea_field($description);
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            update_post_meta($post_id, 'rank_math_description', $description);
            $applied = true;
        }
        
        // All in One SEO
        if (in_array('aioseo', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_aioseop_description', $description);
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_seo_description', $description);
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir canonical URL
     */
    private function set_canonical($post_id, $canonical) {
        $canonical = esc_url($canonical);
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_yoast_wpseo_canonical', $canonical);
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            update_post_meta($post_id, 'rank_math_canonical_url', $canonical);
            $applied = true;
        }
        
        // All in One SEO
        if (in_array('aioseo', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_aioseop_custom_link', $canonical);
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_canonical', $canonical);
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir meta robots
     * 
     * @param int $post_id
     * @param string|array $robots Ex: "noindex,nofollow" ou ["noindex", "nofollow"]
     */
    private function set_robots($post_id, $robots) {
        // Normalizar para string
        if (is_array($robots)) {
            $robots = implode(',', $robots);
        }
        
        $robots = sanitize_text_field($robots);
        $applied = false;
        
        // Parsear diretivas
        $directives = array_map('trim', explode(',', strtolower($robots)));
        
        $noindex = in_array('noindex', $directives);
        $nofollow = in_array('nofollow', $directives);
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            if ($noindex) {
                update_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', '1');
            }
            if ($nofollow) {
                update_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', '1');
            }
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            update_post_meta($post_id, 'rank_math_robots', $directives);
            $applied = true;
        }
        
        // All in One SEO
        if (in_array('aioseo', $this->active_seo_plugins)) {
            if ($noindex) {
                update_post_meta($post_id, '_aioseop_noindex', 'on');
            }
            if ($nofollow) {
                update_post_meta($post_id, '_aioseop_nofollow', 'on');
            }
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_robots', $robots);
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir Open Graph
     */
    private function set_open_graph($post_id, $og_data) {
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            if (!empty($og_data['title'])) {
                update_post_meta($post_id, '_yoast_wpseo_opengraph-title', sanitize_text_field($og_data['title']));
            }
            if (!empty($og_data['description'])) {
                update_post_meta($post_id, '_yoast_wpseo_opengraph-description', sanitize_textarea_field($og_data['description']));
            }
            if (!empty($og_data['image'])) {
                update_post_meta($post_id, '_yoast_wpseo_opengraph-image', esc_url($og_data['image']));
            }
            if (!empty($og_data['type'])) {
                update_post_meta($post_id, '_yoast_wpseo_opengraph-type', sanitize_text_field($og_data['type']));
            }
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            if (!empty($og_data['title'])) {
                update_post_meta($post_id, 'rank_math_facebook_title', sanitize_text_field($og_data['title']));
            }
            if (!empty($og_data['description'])) {
                update_post_meta($post_id, 'rank_math_facebook_description', sanitize_textarea_field($og_data['description']));
            }
            if (!empty($og_data['image'])) {
                update_post_meta($post_id, 'rank_math_facebook_image', esc_url($og_data['image']));
            }
            $applied = true;
        }
        
        // All in One SEO
        if (in_array('aioseo', $this->active_seo_plugins)) {
            if (!empty($og_data['title'])) {
                update_post_meta($post_id, '_aioseop_opengraph_settings_title', sanitize_text_field($og_data['title']));
            }
            if (!empty($og_data['description'])) {
                update_post_meta($post_id, '_aioseop_opengraph_settings_desc', sanitize_textarea_field($og_data['description']));
            }
            if (!empty($og_data['image'])) {
                update_post_meta($post_id, '_aioseop_opengraph_settings_image', esc_url($og_data['image']));
            }
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_og_data', wp_json_encode($og_data));
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir Twitter Cards
     */
    private function set_twitter_cards($post_id, $twitter_data) {
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            if (!empty($twitter_data['title'])) {
                update_post_meta($post_id, '_yoast_wpseo_twitter-title', sanitize_text_field($twitter_data['title']));
            }
            if (!empty($twitter_data['description'])) {
                update_post_meta($post_id, '_yoast_wpseo_twitter-description', sanitize_textarea_field($twitter_data['description']));
            }
            if (!empty($twitter_data['image'])) {
                update_post_meta($post_id, '_yoast_wpseo_twitter-image', esc_url($twitter_data['image']));
            }
            if (!empty($twitter_data['card'])) {
                update_post_meta($post_id, '_yoast_wpseo_twitter_card_type', sanitize_text_field($twitter_data['card']));
            }
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            if (!empty($twitter_data['title'])) {
                update_post_meta($post_id, 'rank_math_twitter_title', sanitize_text_field($twitter_data['title']));
            }
            if (!empty($twitter_data['description'])) {
                update_post_meta($post_id, 'rank_math_twitter_description', sanitize_textarea_field($twitter_data['description']));
            }
            if (!empty($twitter_data['image'])) {
                update_post_meta($post_id, 'rank_math_twitter_image', esc_url($twitter_data['image']));
            }
            if (!empty($twitter_data['card'])) {
                update_post_meta($post_id, 'rank_math_twitter_card_type', sanitize_text_field($twitter_data['card']));
            }
            $applied = true;
        }
        
        // All in One SEO - Suporte limitado
        if (in_array('aioseo', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_aioseop_twitter_data', wp_json_encode($twitter_data));
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_twitter_data', wp_json_encode($twitter_data));
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir focus keyword
     */
    private function set_focus_keyword($post_id, $keyword) {
        $keyword = sanitize_text_field($keyword);
        $applied = false;
        
        // Yoast SEO
        if (in_array('yoast', $this->active_seo_plugins)) {
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $keyword);
            $applied = true;
        }
        
        // Rank Math
        if (in_array('rankmath', $this->active_seo_plugins)) {
            update_post_meta($post_id, 'rank_math_focus_keyword', $keyword);
            $applied = true;
        }
        
        return $applied;
    }
    
    /**
     * Definir FAQ Schema (Yoast/Rank Math)
     */
    private function set_faq_schema($post_id, $schema_data) {
        if (empty($schema_data['mainEntity'])) {
            return false;
        }
        
        $applied = false;
        
        // Rank Math - Suporte nativo a FAQ
        if (in_array('rankmath', $this->active_seo_plugins)) {
            $faq_items = array();
            
            foreach ($schema_data['mainEntity'] as $item) {
                if (!empty($item['name']) && !empty($item['acceptedAnswer']['text'])) {
                    $faq_items[] = array(
                        'question' => sanitize_text_field($item['name']),
                        'answer' => wp_kses_post($item['acceptedAnswer']['text']),
                    );
                }
            }
            
            if (!empty($faq_items)) {
                update_post_meta($post_id, 'rank_math_faq_data', $faq_items);
                update_post_meta($post_id, 'rank_math_rich_snippet', 'faq');
                $applied = true;
            }
        }
        
        // Yoast SEO - Schema blocks (limitado)
        if (in_array('yoast', $this->active_seo_plugins)) {
            // Yoast usa blocos Gutenberg, armazenar JSON genérico
            update_post_meta($post_id, '_sifet_faq_schema', wp_json_encode($schema_data));
            $applied = true;
        }
        
        // Fallback genérico
        if (!$applied) {
            update_post_meta($post_id, '_sifet_schema_faq', wp_json_encode($schema_data));
            $applied = true;
        }
        
        error_log("[Sifet SEO] FAQ Schema aplicado ao post {$post_id}");
        return $applied;
    }
    
    /**
     * Definir HowTo Schema
     */
    private function set_howto_schema($post_id, $schema_data) {
        if (empty($schema_data['step'])) {
            return false;
        }
        
        $applied = false;
        
        // Rank Math - Suporte nativo a HowTo
        if (in_array('rankmath', $this->active_seo_plugins)) {
            $steps = array();
            
            foreach ($schema_data['step'] as $step) {
                if (!empty($step['name'])) {
                    $steps[] = array(
                        'name' => sanitize_text_field($step['name']),
                        'text' => !empty($step['text']) ? wp_kses_post($step['text']) : '',
                        'image' => !empty($step['image']) ? esc_url($step['image']) : '',
                    );
                }
            }
            
            if (!empty($steps)) {
                update_post_meta($post_id, 'rank_math_howto_data', $steps);
                update_post_meta($post_id, 'rank_math_rich_snippet', 'howto');
                
                // Dados adicionais
                if (!empty($schema_data['totalTime'])) {
                    update_post_meta($post_id, 'rank_math_howto_total_time', sanitize_text_field($schema_data['totalTime']));
                }
                
                $applied = true;
            }
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_schema_howto', wp_json_encode($schema_data));
            $applied = true;
        }
        
        error_log("[Sifet SEO] HowTo Schema aplicado ao post {$post_id}");
        return $applied;
    }
    
    /**
     * Definir Review Schema
     */
    private function set_review_schema($post_id, $schema_data) {
        $applied = false;
        
        // Rank Math - Suporte a Review
        if (in_array('rankmath', $this->active_seo_plugins)) {
            if (!empty($schema_data['reviewRating']['ratingValue'])) {
                update_post_meta($post_id, 'rank_math_review_rating_value', floatval($schema_data['reviewRating']['ratingValue']));
            }
            
            if (!empty($schema_data['reviewRating']['bestRating'])) {
                update_post_meta($post_id, 'rank_math_review_rating_max', floatval($schema_data['reviewRating']['bestRating']));
            }
            
            if (!empty($schema_data['reviewRating']['worstRating'])) {
                update_post_meta($post_id, 'rank_math_review_rating_min', floatval($schema_data['reviewRating']['worstRating']));
            }
            
            update_post_meta($post_id, 'rank_math_rich_snippet', 'review');
            $applied = true;
        }
        
        // Fallback
        if (!$applied) {
            update_post_meta($post_id, '_sifet_schema_review', wp_json_encode($schema_data));
            $applied = true;
        }
        
        error_log("[Sifet SEO] Review Schema aplicado ao post {$post_id}");
        return $applied;
    }
    
    /**
     * Schema genérico
     */
    private function set_generic_schema($post_id, $schema_data) {
        update_post_meta($post_id, '_sifet_schema_generic', wp_json_encode($schema_data));
        error_log("[Sifet SEO] Schema genérico aplicado ao post {$post_id}");
        return true;
    }
    
    /**
     * Obter metadados SEO de um post
     * 
     * @param int $post_id
     * @return array Dados SEO
     */
    public function get_seo_metadata($post_id) {
        $data = array();
        
        // Detectar plugin ativo e buscar metadados
        if (in_array('yoast', $this->active_seo_plugins)) {
            $data['title'] = get_post_meta($post_id, '_yoast_wpseo_title', true);
            $data['description'] = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
            $data['canonical'] = get_post_meta($post_id, '_yoast_wpseo_canonical', true);
            $data['plugin'] = 'yoast';
        } elseif (in_array('rankmath', $this->active_seo_plugins)) {
            $data['title'] = get_post_meta($post_id, 'rank_math_title', true);
            $data['description'] = get_post_meta($post_id, 'rank_math_description', true);
            $data['canonical'] = get_post_meta($post_id, 'rank_math_canonical_url', true);
            $data['plugin'] = 'rankmath';
        } elseif (in_array('aioseo', $this->active_seo_plugins)) {
            $data['title'] = get_post_meta($post_id, '_aioseop_title', true);
            $data['description'] = get_post_meta($post_id, '_aioseop_description', true);
            $data['canonical'] = get_post_meta($post_id, '_aioseop_custom_link', true);
            $data['plugin'] = 'aioseo';
        } else {
            // Fallback
            $data['title'] = get_post_meta($post_id, '_sifet_seo_title', true);
            $data['description'] = get_post_meta($post_id, '_sifet_seo_description', true);
            $data['canonical'] = get_post_meta($post_id, '_sifet_canonical', true);
            $data['plugin'] = 'none';
        }
        
        return array_filter($data); // Remove vazios
    }
}

// Inicializar
global $sifet_seo_manager;
$sifet_seo_manager = new Sifet_SEO_Manager();
