<?php
/**
 * Sifet Media Manager
 * 
 * Gerenciamento de mídia com deduplicação, conversão WebP e processamento assíncrono
 * 
 * @package Sifet
 * @version 1.0.8
 */

if (!defined('ABSPATH')) {
    exit;
}

class Sifet_Media_Manager {
    
    /**
     * Meta key para hash da URL de origem
     */
    const META_SOURCE_HASH = '_sifet_src_hash';
    
    /**
     * Meta key para URL original
     */
    const META_SOURCE_URL = '_sifet_src_url';
    
    /**
     * Meta key para indicar se foi convertido para WebP
     */
    const META_WEBP_CONVERTED = '_sifet_webp_converted';
    
    /**
     * Timeout padrão para download (segundos)
     */
    const DEFAULT_TIMEOUT = 5;
    
    /**
     * Número de tentativas de retry
     */
    const MAX_RETRIES = 2;
    
    /**
     * Content-Types válidos
     */
    const VALID_CONTENT_TYPES = array(
        'image/jpeg',
        'image/jpg',
        'image/png',
        'image/gif',
        'image/webp',
        'image/svg+xml',
    );
    
    /**
     * Inicializar hooks
     */
    public function __construct() {
        // Hook para Action Scheduler (se disponível)
        add_action('sifet_process_media_queue', array($this, 'process_media_queue'), 10, 1);
    }
    
    /**
     * Fazer upload de imagem com deduplicação
     * 
     * @param int $post_id ID do post
     * @param string $image_url URL da imagem
     * @param string $filename Nome do arquivo
     * @param string $alt_text Texto alternativo
     * @param string $title Título (fallback: título do post)
     * @return int|WP_Error ID do attachment ou erro
     */
    public function upload_image($post_id, $image_url, $filename = '', $alt_text = '', $title = '') {
        // Validar URL
        if (empty($image_url) || !filter_var($image_url, FILTER_VALIDATE_URL)) {
            return new WP_Error(
                'SIFET_InvalidURL',
                __('URL da imagem inválida.', 'sifet')
            );
        }
        
        // Gerar hash da URL para deduplicação
        $url_hash = md5($image_url);
        
        // Verificar se imagem já existe
        $existing_attachment = $this->find_existing_attachment($url_hash);
        if ($existing_attachment) {
            error_log("[Sifet Media] Imagem já existe (dedupe): attachment_id={$existing_attachment}");
            
            // Atualizar alt text se fornecido
            if (!empty($alt_text)) {
                update_post_meta($existing_attachment, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
            }
            
            return $existing_attachment;
        }
        
        // Fazer download com retry
        $download_result = $this->download_image_with_retry($image_url);
        
        if (is_wp_error($download_result)) {
            return $download_result;
        }
        
        // Validar Content-Type
        $content_type = $download_result['content_type'];
        if (!$this->is_valid_content_type($content_type)) {
            return new WP_Error(
                'SIFET_InvalidContentType',
                sprintf(
                    __('Content-Type inválido: %s. Esperado: image/*', 'sifet'),
                    $content_type
                )
            );
        }
        
        // Processar filename
        if (empty($filename)) {
            $filename = basename(parse_url($image_url, PHP_URL_PATH));
        }
        $filename = sanitize_file_name($filename);
        
        // Adicionar extensão se ausente
        $wp_filetype = wp_check_filetype($filename, null);
        if (empty($wp_filetype['ext'])) {
            $extension = $this->get_extension_from_content_type($content_type);
            $filename .= '.' . $extension;
        }
        
        // Salvar imagem
        $upload_dir = wp_upload_dir();
        
        if (wp_mkdir_p($upload_dir['path'])) {
            $filename = wp_unique_filename($upload_dir['path'], $filename);
            $file_path = $upload_dir['path'] . '/' . $filename;
        } else {
            $filename = wp_unique_filename($upload_dir['basedir'], $filename);
            $file_path = $upload_dir['basedir'] . '/' . $filename;
        }
        
        // Escrever arquivo
        $bytes_written = file_put_contents($file_path, $download_result['data']);
        
        if ($bytes_written === false) {
            return new WP_Error(
                'SIFET_FileWriteError',
                __('Erro ao salvar imagem no servidor.', 'sifet')
            );
        }
        
        // Criar attachment
        $attachment_id = $this->create_attachment($file_path, $post_id, $filename);
        
        if (is_wp_error($attachment_id)) {
            @unlink($file_path); // Limpar arquivo em caso de erro
            return $attachment_id;
        }
        
        // Salvar metadados
        update_post_meta($attachment_id, self::META_SOURCE_HASH, $url_hash);
        update_post_meta($attachment_id, self::META_SOURCE_URL, esc_url($image_url));
        
        // Alt text e title
        $this->set_attachment_metadata($attachment_id, $alt_text, $title, $post_id);
        
        // Converter para WebP se habilitado
        if (get_option('sifet_convert_webp', false)) {
            $this->convert_to_webp($attachment_id, $file_path);
        }
        
        return $attachment_id;
    }
    
    /**
     * Fazer download da imagem com retry
     * 
     * @param string $url URL da imagem
     * @return array|WP_Error Array com 'data' e 'content_type' ou erro
     */
    private function download_image_with_retry($url) {
        $last_error = null;
        
        for ($attempt = 1; $attempt <= self::MAX_RETRIES; $attempt++) {
            $result = $this->download_image($url);
            
            if (!is_wp_error($result)) {
                if ($attempt > 1) {
                    error_log("[Sifet Media] Download bem-sucedido após {$attempt} tentativas");
                }
                return $result;
            }
            
            $last_error = $result;
            error_log("[Sifet Media] Tentativa {$attempt}/{" . self::MAX_RETRIES . "} falhou: " . $result->get_error_message());
            
            // Aguardar antes de retry (exponential backoff)
            if ($attempt < self::MAX_RETRIES) {
                usleep(pow(2, $attempt) * 100000); // 200ms, 400ms
            }
        }
        
        return new WP_Error(
            'SIFET_MediaTimeout',
            sprintf(
                __('Timeout ao fazer download da imagem após %d tentativas.', 'sifet'),
                self::MAX_RETRIES
            )
        );
    }
    
    /**
     * Fazer download da imagem
     * 
     * @param string $url URL da imagem
     * @return array|WP_Error Array com 'data' e 'content_type' ou erro
     */
    private function download_image($url) {
        $response = wp_remote_get($url, array(
            'timeout' => self::DEFAULT_TIMEOUT,
            'redirection' => 5,
            'user-agent' => 'Sifet-WordPress-Plugin/1.0',
            'sslverify' => true,
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code !== 200) {
            return new WP_Error(
                'SIFET_HTTPError',
                sprintf(
                    __('Erro HTTP %d ao fazer download da imagem.', 'sifet'),
                    $status_code
                )
            );
        }
        
        $data = wp_remote_retrieve_body($response);
        
        if (empty($data)) {
            return new WP_Error(
                'SIFET_EmptyResponse',
                __('Resposta vazia ao fazer download da imagem.', 'sifet')
            );
        }
        
        $content_type = wp_remote_retrieve_header($response, 'content-type');
        
        // Remover charset se presente (image/jpeg; charset=utf-8 -> image/jpeg)
        $content_type = strtok($content_type, ';');
        
        return array(
            'data' => $data,
            'content_type' => $content_type,
        );
    }
    
    /**
     * Verificar se Content-Type é válido
     * 
     * @param string $content_type Content-Type
     * @return bool
     */
    private function is_valid_content_type($content_type) {
        return in_array(strtolower($content_type), self::VALID_CONTENT_TYPES, true);
    }
    
    /**
     * Obter extensão a partir do Content-Type
     * 
     * @param string $content_type Content-Type
     * @return string Extensão
     */
    private function get_extension_from_content_type($content_type) {
        $map = array(
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/svg+xml' => 'svg',
        );
        
        return isset($map[$content_type]) ? $map[$content_type] : 'jpg';
    }
    
    /**
     * Criar attachment no WordPress
     * 
     * @param string $file_path Caminho do arquivo
     * @param int $post_id ID do post
     * @param string $filename Nome do arquivo
     * @return int|WP_Error ID do attachment ou erro
     */
    private function create_attachment($file_path, $post_id, $filename) {
        $wp_filetype = wp_check_filetype($filename, null);
        
        $attachment = array(
            'post_mime_type' => $wp_filetype['type'],
            'post_title' => sanitize_file_name(pathinfo($filename, PATHINFO_FILENAME)),
            'post_content' => '',
            'post_status' => 'inherit',
        );
        
        $attachment_id = wp_insert_attachment($attachment, $file_path, $post_id);
        
        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }
        
        // Gerar metadados
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attachment_data = wp_generate_attachment_metadata($attachment_id, $file_path);
        wp_update_attachment_metadata($attachment_id, $attachment_data);
        
        return $attachment_id;
    }
    
    /**
     * Definir metadados do attachment (alt, title)
     * 
     * @param int $attachment_id ID do attachment
     * @param string $alt_text Alt text
     * @param string $title Title
     * @param int $post_id ID do post (para fallback)
     */
    private function set_attachment_metadata($attachment_id, $alt_text, $title, $post_id) {
        // Alt text
        if (!empty($alt_text)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
        } else {
            // Fallback: título do post
            $post = get_post($post_id);
            if ($post) {
                update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($post->post_title));
            }
        }
        
        // Title
        if (!empty($title)) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_title' => sanitize_text_field($title),
            ));
        } else {
            // Fallback: título do post
            $post = get_post($post_id);
            if ($post) {
                wp_update_post(array(
                    'ID' => $attachment_id,
                    'post_title' => sanitize_text_field($post->post_title),
                ));
            }
        }
    }
    
    /**
     * Converter imagem para WebP
     * 
     * @param int $attachment_id ID do attachment
     * @param string $file_path Caminho do arquivo original
     * @return bool True se convertido, false se falhou
     */
    private function convert_to_webp($attachment_id, $file_path) {
        // Verificar se GD ou Imagick suporta WebP
        if (!$this->webp_supported()) {
            error_log('[Sifet Media] WebP não suportado pelo servidor');
            return false;
        }
        
        // Verificar se já foi convertido
        if (get_post_meta($attachment_id, self::META_WEBP_CONVERTED, true)) {
            return false;
        }
        
        try {
            $webp_path = preg_replace('/\.(jpe?g|png|gif)$/i', '.webp', $file_path);
            
            // Tentar converter
            $converted = false;
            
            if (function_exists('imagewebp')) {
                $converted = $this->convert_with_gd($file_path, $webp_path);
            } elseif (class_exists('Imagick')) {
                $converted = $this->convert_with_imagick($file_path, $webp_path);
            }
            
            if ($converted && file_exists($webp_path)) {
                // Criar novo attachment para WebP
                $webp_attachment_id = $this->create_attachment(
                    $webp_path,
                    wp_get_post_parent_id($attachment_id),
                    basename($webp_path)
                );
                
                if (!is_wp_error($webp_attachment_id)) {
                    // Marcar ambos
                    update_post_meta($attachment_id, self::META_WEBP_CONVERTED, $webp_attachment_id);
                    update_post_meta($webp_attachment_id, '_sifet_webp_original', $attachment_id);
                    
                    error_log("[Sifet Media] Imagem convertida para WebP: {$attachment_id} -> {$webp_attachment_id}");
                    return true;
                }
            }
        } catch (Exception $e) {
            error_log('[Sifet Media] Erro ao converter para WebP: ' . $e->getMessage());
        }
        
        return false;
    }
    
    /**
     * Verificar se WebP é suportado
     * 
     * @return bool
     */
    private function webp_supported() {
        if (function_exists('imagewebp')) {
            return true;
        }
        
        if (class_exists('Imagick')) {
            $imagick = new Imagick();
            return in_array('WEBP', $imagick->queryFormats(), true);
        }
        
        return false;
    }
    
    /**
     * Converter imagem com GD
     * 
     * @param string $source Caminho de origem
     * @param string $destination Caminho de destino
     * @return bool
     */
    private function convert_with_gd($source, $destination) {
        $info = getimagesize($source);
        
        if ($info === false) {
            return false;
        }
        
        $image = null;
        
        switch ($info['mime']) {
            case 'image/jpeg':
                $image = imagecreatefromjpeg($source);
                break;
            case 'image/png':
                $image = imagecreatefrompng($source);
                // Preservar transparência
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
                break;
            case 'image/gif':
                $image = imagecreatefromgif($source);
                break;
        }
        
        if ($image === null) {
            return false;
        }
        
        $quality = (int) get_option('sifet_webp_quality', 80);
        $result = imagewebp($image, $destination, $quality);
        
        imagedestroy($image);
        
        return $result;
    }
    
    /**
     * Converter imagem com Imagick
     * 
     * @param string $source Caminho de origem
     * @param string $destination Caminho de destino
     * @return bool
     */
    private function convert_with_imagick($source, $destination) {
        try {
            $imagick = new Imagick($source);
            $imagick->setImageFormat('webp');
            
            $quality = (int) get_option('sifet_webp_quality', 80);
            $imagick->setImageCompressionQuality($quality);
            
            $result = $imagick->writeImage($destination);
            $imagick->clear();
            $imagick->destroy();
            
            return $result;
        } catch (Exception $e) {
            error_log('[Sifet Media] Imagick error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Buscar attachment existente pelo hash da URL
     * 
     * @param string $url_hash Hash MD5 da URL
     * @return int|false ID do attachment ou false
     */
    private function find_existing_attachment($url_hash) {
        global $wpdb;
        
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
             WHERE meta_key = %s AND meta_value = %s 
             LIMIT 1",
            self::META_SOURCE_HASH,
            $url_hash
        ));
        
        if ($attachment_id) {
            // Verificar se attachment ainda existe
            $post = get_post($attachment_id);
            if ($post && $post->post_type === 'attachment') {
                return (int) $attachment_id;
            }
        }
        
        return false;
    }
    
    /**
     * Processar fila de imagens (Action Scheduler)
     * 
     * @param array $items Array de itens: ['post_id', 'url', 'filename', 'alt', 'title']
     */
    public function process_media_queue($items) {
        if (!is_array($items) || empty($items)) {
            return;
        }
        
        error_log('[Sifet Media] Processando fila de ' . count($items) . ' imagens');
        
        foreach ($items as $item) {
            $result = $this->upload_image(
                $item['post_id'],
                $item['url'],
                $item['filename'] ?? '',
                $item['alt'] ?? '',
                $item['title'] ?? ''
            );
            
            if (is_wp_error($result)) {
                error_log('[Sifet Media] Erro na fila: ' . $result->get_error_message());
            } else {
                error_log("[Sifet Media] Imagem processada: attachment_id={$result}");
            }
        }
    }
    
    /**
     * Agendar processamento assíncrono de imagens
     * 
     * @param array $items Array de itens para processar
     * @return bool True se agendado, false se não disponível
     */
    public function schedule_media_processing($items) {
        // Verificar se Action Scheduler está disponível
        if (!function_exists('as_schedule_single_action')) {
            error_log('[Sifet Media] Action Scheduler não disponível, processando sincronicamente');
            $this->process_media_queue($items);
            return false;
        }
        
        // Agendar para daqui a 10 segundos
        as_schedule_single_action(
            time() + 10,
            'sifet_process_media_queue',
            array($items),
            'sifet-media'
        );
        
        error_log('[Sifet Media] Fila agendada com ' . count($items) . ' imagens');
        return true;
    }
    
    /**
     * Limpar cache de deduplicação (manutenção)
     * Remove hashes de attachments que não existem mais
     */
    public function cleanup_orphaned_hashes() {
        global $wpdb;
        
        $orphaned = $wpdb->query(
            "DELETE pm FROM {$wpdb->postmeta} pm
             LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE pm.meta_key = '" . self::META_SOURCE_HASH . "'
             AND p.ID IS NULL"
        );
        
        if ($orphaned > 0) {
            error_log("[Sifet Media] Limpeza: {$orphaned} hashes órfãos removidos");
        }
        
        return $orphaned;
    }
}

// Inicializar
global $sifet_media_manager;
$sifet_media_manager = new Sifet_Media_Manager();
