<?php
/**
 * Painel administrativo do Sifet
 *
 * @package Sifet
 */

if (!defined('ABSPATH')) {
    exit;
}

class Sifet_Admin {
    /**
     * Inicializa hooks de administração.
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'register_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_filter('plugin_action_links_' . SIFET_PLUGIN_BASENAME, array($this, 'add_plugin_action_links'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_notices', array($this, 'display_settings_saved_notice'));
    }

    /**
     * Registra página de configurações.
     */
    public function register_menu() {
        add_options_page(
            __('Sifet', 'sifet'),
            __('Sifet', 'sifet'),
            'manage_options',
            'sifet',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Registra settings e campos.
     */
    public function register_settings() {
        register_setting(
            'sifet_settings',
            'token_sifet',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_token'),
                'default' => '',
            )
        );

        register_setting(
            'sifet_settings',
            'sifet_convert_webp',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false,
            )
        );

        register_setting(
            'sifet_settings',
            'sifet_webp_quality',
            array(
                'type' => 'integer',
                'sanitize_callback' => array($this, 'sanitize_quality'),
                'default' => 80,
            )
        );

        register_setting(
            'sifet_settings',
            'sifet_default_author',
            array(
                'type' => 'integer',
                'sanitize_callback' => array($this, 'sanitize_author_id'),
                'default' => 0,
            )
        );

        add_settings_section(
            'sifet_section_token',
            __('Autenticação', 'sifet'),
            array($this, 'render_token_section'),
            'sifet'
        );

        add_settings_field(
            'token_sifet',
            __('Token de acesso', 'sifet'),
            array($this, 'render_token_field'),
            'sifet',
            'sifet_section_token'
        );

        add_settings_section(
            'sifet_section_content',
            __('Conteúdo', 'sifet'),
            array($this, 'render_content_section'),
            'sifet'
        );

        add_settings_field(
            'sifet_default_author',
            __('Autor padrão dos posts', 'sifet'),
            array($this, 'render_default_author_field'),
            'sifet',
            'sifet_section_content'
        );

        add_settings_section(
            'sifet_section_media',
            __('Mídia', 'sifet'),
            array($this, 'render_media_section'),
            'sifet'
        );

        add_settings_field(
            'sifet_convert_webp',
            __('Converter imagens para WebP', 'sifet'),
            array($this, 'render_convert_webp_field'),
            'sifet',
            'sifet_section_media'
        );

        add_settings_field(
            'sifet_webp_quality',
            __('Qualidade do WebP (0-100)', 'sifet'),
            array($this, 'render_quality_field'),
            'sifet',
            'sifet_section_media'
        );
    }

    /**
     * Link rápido nas ações do plugin.
     *
     * @param array $links Links existentes.
     * @return array
     */
    public function add_plugin_action_links($links) {
        $url = admin_url('options-general.php?page=sifet');
        $links[] = sprintf(
            '<a href="%s">%s</a>',
            esc_url($url),
            esc_html__('Configurações', 'sifet')
        );

        return $links;
    }

    /**
     * Enfileira estilos para a página do plugin.
     *
     * @param string $hook Hook atual.
     */
    public function enqueue_assets($hook) {
        if ($hook !== 'settings_page_sifet') {
            return;
        }

        wp_enqueue_style(
            'sifet-admin',
            SIFET_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SIFET_PLUGIN_VERSION
        );
    }

    /**
     * Exibe mensagem personalizada quando as configurações são salvas.
     */
    public function display_settings_saved_notice() {
        // Verifica se estamos na página do plugin
        $screen = get_current_screen();
        if ($screen->id !== 'settings_page_sifet') {
            return;
        }

        // Verifica se as configurações foram atualizadas
        if (isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true') {
            ?>
            <div class="notice notice-success is-dismissible sifet-notice">
                <p>
                    <strong><?php echo esc_html__('Configurações salvas com sucesso!', 'sifet'); ?></strong>
                    <?php echo esc_html__('Suas preferências da Sifet foram atualizadas.', 'sifet'); ?>
                </p>
            </div>
            <style>
                .sifet-notice {
                    border-left-color: #3B82F6;
                }
                .sifet-notice p {
                    margin: 0.5em 0;
                }
            </style>
            <?php
        }
    }

    /**
     * Conteúdo do grupo de token.
     */
    public function render_token_section() {
        echo '<p>' . esc_html__('Informe o token fornecido pela Sifet para autenticar as requisições.', 'sifet') . '</p>';
    }

    /**
     * Campo de token.
     */
    public function render_token_field() {
        $token = get_option('token_sifet', '');
        printf(
            '<input type="text" class="regular-text" id="token_sifet" name="token_sifet" value="%s" autocomplete="off" />',
            esc_attr($token)
        );
    }

    /**
     * Conteúdo da seção de mídia.
     */
    public function render_media_section() {
        echo '<p>' . esc_html__('Opções aplicadas apenas quando o gerenciador de mídia da Sifet é utilizado.', 'sifet') . '</p>';
    }

    /**
     * Campo de toggle WebP.
     */
    public function render_convert_webp_field() {
        $enabled = (bool) get_option('sifet_convert_webp', false);
        printf(
            '<label><input type="checkbox" id="sifet_convert_webp" name="sifet_convert_webp" value="1" %s /> %s</label>',
            checked($enabled, true, false),
            esc_html__('Ativar conversão automática para WebP (quando suportado).', 'sifet')
        );
    }

    /**
     * Campo de qualidade.
     */
    public function render_quality_field() {
        $quality = (int) get_option('sifet_webp_quality', 80);
        printf(
            '<input type="number" min="0" max="100" step="1" id="sifet_webp_quality" name="sifet_webp_quality" value="%d" />',
            $quality
        );
    }

    /**
     * Conteúdo da seção de conteúdo.
     */
    public function render_content_section() {
        echo '<p>' . esc_html__('Configurações relacionadas à criação de posts.', 'sifet') . '</p>';
    }

    /**
     * Campo de autor padrão.
     */
    public function render_default_author_field() {
        $selected = (int) get_option('sifet_default_author', 0);
        
        // Se nenhum autor estiver salvo, tentar pegar o usuário atual
        if (empty($selected) && is_user_logged_in()) {
            $selected = get_current_user_id();
        }

        $users = get_users(array(
            'who' => 'authors',
            'orderby' => 'display_name',
            'order' => 'ASC',
            'fields' => array('ID', 'display_name', 'user_email')
        ));
        
        echo '<div class="sifet-select-wrapper">';
        echo '<select id="sifet_default_author" name="sifet_default_author" class="sifet-field__select">';
        echo '<option value="0">' . esc_html__('Selecione um autor...', 'sifet') . '</option>';
        
        foreach ($users as $user) {
            printf(
                '<option value="%d" %s>%s (%s)</option>',
                $user->ID,
                selected($selected, $user->ID, false),
                esc_html($user->display_name),
                esc_html($user->user_email)
            );
        }
        
        echo '</select>';
        echo '<div class="sifet-select-arrow">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" d="M19.5 8.25l-7.5 7.5-7.5-7.5" />
            </svg>
        </div>';
        echo '</div>';
    }

    /**
     * Página de configurações.
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $token = get_option('token_sifet', '');
        $token_status = !empty($token);
        ?>
        <div class="wrap sifet-admin">
            <!-- Header -->
            <div class="sifet-header">
                <div class="sifet-header__content">
                    <div class="sifet-header__brand">
                        <img src="<?php echo esc_url(SIFET_PLUGIN_URL . 'assets/images/logo.png'); ?>" alt="<?php echo esc_attr__('Sifet', 'sifet'); ?>" class="sifet-header__logo" />
                        <div>
                            <h1 class="sifet-header__title"><?php echo esc_html__('Configurações da Sifet', 'sifet'); ?></h1>
                            <p class="sifet-header__subtitle"><?php echo esc_html__('Integração WordPress', 'sifet'); ?></p>
                        </div>
                    </div>
                    <div class="sifet-status">
                        <span class="sifet-status__label"><?php echo esc_html__('Status:', 'sifet'); ?></span>
                        <span class="sifet-status__badge sifet-status__badge--<?php echo $token_status ? 'active' : 'inactive'; ?>">
                            <span class="sifet-status__dot"></span>
                            <?php echo $token_status ? esc_html__('Conectado', 'sifet') : esc_html__('Desconectado', 'sifet'); ?>
                        </span>
                    </div>
                </div>
            </div>

            <form method="post" action="options.php" class="sifet-form">
                <?php settings_fields('sifet_settings'); ?>

                <div class="sifet-grid">
                    <!-- Card de Autenticação -->
                    <div class="sifet-card">
                        <div class="sifet-card__header">
                            <div class="sifet-card__icon sifet-card__icon--primary">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M15.75 5.25a3 3 0 013 3m3 0a6 6 0 01-7.029 5.912c-.563-.097-1.159.026-1.563.43L10.5 17.25H8.25v2.25H6v2.25H2.25v-2.818c0-.597.237-1.17.659-1.591l6.499-6.499c.404-.404.527-1 .43-1.563A6 6 0 1121.75 8.25z" />
                                </svg>
                            </div>
                            <div>
                                <h2 class="sifet-card__title"><?php echo esc_html__('Autenticação', 'sifet'); ?></h2>
                                <p class="sifet-card__description"><?php echo esc_html__('Configure o token de acesso à API', 'sifet'); ?></p>
                            </div>
                        </div>

                        <div class="sifet-card__body">
                            <div class="sifet-field">
                                <label for="token_sifet" class="sifet-field__label">
                                    <?php echo esc_html__('Token de acesso', 'sifet'); ?>
                                    <span class="sifet-field__required">*</span>
                                </label>
                                <input
                                    type="text"
                                    id="token_sifet"
                                    name="token_sifet"
                                    class="sifet-field__input"
                                    value="<?php echo esc_attr($token); ?>"
                                    placeholder="<?php echo esc_attr__('Cole seu token aqui...', 'sifet'); ?>"
                                    autocomplete="off"
                                />
                                <p class="sifet-field__help">
                                    <?php echo esc_html__('Informe o token fornecido pela Sifet para autenticar as requisições da API.', 'sifet'); ?>
                                </p>
                            </div>
                        </div>
                    </div>

                    <!-- Card de Configurações de Conteúdo -->
                    <div class="sifet-card">
                        <div class="sifet-card__header">
                            <div class="sifet-card__icon sifet-card__icon--primary">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M16.862 4.487l1.687-1.688a1.875 1.875 0 112.652 2.652L10.582 16.07a4.5 4.5 0 01-1.897 1.13L6 18l.8-2.685a4.5 4.5 0 011.13-1.897l8.932-8.931zm0 0L19.5 7.125M18 14v4.75A2.25 2.25 0 0115.75 21H5.25A2.25 2.25 0 013 18.75V8.25A2.25 2.25 0 015.25 6H10" />
                                </svg>
                            </div>
                            <div>
                                <h2 class="sifet-card__title"><?php echo esc_html__('Conteúdo', 'sifet'); ?></h2>
                                <p class="sifet-card__description"><?php echo esc_html__('Personalize como os posts são criados', 'sifet'); ?></p>
                            </div>
                        </div>

                        <div class="sifet-card__body">
                            <div class="sifet-field">
                                <label for="sifet_default_author" class="sifet-field__label">
                                    <?php echo esc_html__('Autor padrão dos posts', 'sifet'); ?>
                                </label>
                                <?php $this->render_default_author_field(); ?>
                                <p class="sifet-field__help">
                                    <?php echo esc_html__('Este usuário será atribuído como autor dos posts importados via API.', 'sifet'); ?>
                                </p>
                            </div>
                        </div>
                    </div>

                    <!-- Card de Configurações de Mídia -->
                    <div class="sifet-card">
                        <div class="sifet-card__header">
                            <div class="sifet-card__icon sifet-card__icon--primary">
                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M2.25 15.75l5.159-5.159a2.25 2.25 0 013.182 0l5.159 5.159m-1.5-1.5l1.409-1.409a2.25 2.25 0 013.182 0l2.909 2.909m-18 3.75h16.5a1.5 1.5 0 001.5-1.5V6a1.5 1.5 0 00-1.5-1.5H3.75A1.5 1.5 0 002.25 6v12a1.5 1.5 0 001.5 1.5zm10.5-11.25h.008v.008h-.008V8.25zm.375 0a.375.375 0 11-.75 0 .375.375 0 01.75 0z" />
                                </svg>
                            </div>
                            <div>
                                <h2 class="sifet-card__title"><?php echo esc_html__('Configurações de Mídia', 'sifet'); ?></h2>
                                <p class="sifet-card__description"><?php echo esc_html__('Otimize suas imagens automaticamente', 'sifet'); ?></p>
                            </div>
                        </div>

                        <div class="sifet-card__body">
                            <div class="sifet-field">
                                <div class="sifet-toggle">
                                    <input
                                        type="checkbox"
                                        id="sifet_convert_webp"
                                        name="sifet_convert_webp"
                                        class="sifet-toggle__input"
                                        value="1"
                                        <?php checked((bool) get_option('sifet_convert_webp', false), true); ?>
                                    />
                                    <label for="sifet_convert_webp" class="sifet-toggle__label">
                                        <span class="sifet-toggle__switch"></span>
                                        <span class="sifet-toggle__text">
                                            <span class="sifet-toggle__title"><?php echo esc_html__('Converter imagens para WebP', 'sifet'); ?></span>
                                            <span class="sifet-toggle__description"><?php echo esc_html__('Converte automaticamente imagens para formato WebP quando suportado', 'sifet'); ?></span>
                                        </span>
                                    </label>
                                </div>
                            </div>

                            <div class="sifet-field">
                                <label for="sifet_webp_quality" class="sifet-field__label">
                                    <?php echo esc_html__('Qualidade do WebP', 'sifet'); ?>
                                </label>
                                <div class="sifet-range">
                                    <input
                                        type="range"
                                        id="sifet_webp_quality"
                                        name="sifet_webp_quality"
                                        class="sifet-range__input"
                                        min="0"
                                        max="100"
                                        step="1"
                                        value="<?php echo esc_attr(get_option('sifet_webp_quality', 80)); ?>"
                                    />
                                    <div class="sifet-range__labels">
                                        <span><?php echo esc_html__('Baixa', 'sifet'); ?></span>
                                        <span class="sifet-range__value"><span id="quality_value"><?php echo esc_html(get_option('sifet_webp_quality', 80)); ?></span>%</span>
                                        <span><?php echo esc_html__('Alta', 'sifet'); ?></span>
                                    </div>
                                </div>
                                <p class="sifet-field__help">
                                    <?php echo esc_html__('Defina a qualidade de compressão para imagens WebP (recomendado: 80-90)', 'sifet'); ?>
                                </p>
                            </div>
                        </div>
                    </div>


                </div>

                <div class="sifet-footer">
                    <button type="submit" class="sifet-button sifet-button--primary">
                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M4.5 12.75l6 6 9-13.5" />
                        </svg>
                        <?php echo esc_html__('Salvar configurações', 'sifet'); ?>
                    </button>
                    <p class="sifet-footer__help">
                        <?php echo esc_html__('As alterações serão aplicadas imediatamente após salvar.', 'sifet'); ?>
                    </p>
                </div>
            </form>
        </div>

        <script>
        (function() {
            const rangeInput = document.getElementById('sifet_webp_quality');
            const valueDisplay = document.getElementById('quality_value');

            if (rangeInput && valueDisplay) {
                rangeInput.addEventListener('input', function() {
                    valueDisplay.textContent = this.value;
                });
            }
        })();
        </script>
        <?php
    }

    /**
     * Sanitiza token.
     *
     * @param string $value Valor recebido.
     * @return string
     */
    public function sanitize_token($value) {
        $value = trim((string) $value);
        return sanitize_text_field($value);
    }

    /**
     * Sanitiza checkbox bool.
     *
     * @param mixed $value Valor recebido.
     * @return bool
     */
    public function sanitize_checkbox($value) {
        return !empty($value);
    }

    /**
     * Sanitiza qualidade.
     *
     * @param mixed $value Valor recebido.
     * @return int
     */
    public function sanitize_quality($value) {
        $value = (int) $value;
        if ($value < 0) {
            $value = 0;
        }
        if ($value > 100) {
            $value = 100;
        }

        return $value;
    }

    /**
     * Sanitiza ID do autor.
     *
     * @param mixed $value Valor recebido.
     * @return int
     */
    public function sanitize_author_id($value) {
        return (int) $value;
    }
}
